// ==UserScript==
// @name         5ch Twitter/X 埋め込み表示（公式widgets.js使用・開閉ボタン付き）
// @namespace    https://example.com/
// @version      1.0.3
// @description  5ch投稿内のTwitter/X URLを開閉ボタン付きで公式埋め込み表示。初期非表示、クリックで表示/非表示切替可能。表示領域に入るまで埋め込み作成を遅延させ縦ズレ軽減。
// @match        *://*.5ch.net/test/read.cgi/*
// @grant        none
// ==/UserScript==

(function () {
    'use strict';

    function loadTwitterWidgets() {
        return new Promise((resolve) => {
            if (window.twttr && window.twttr.widgets) {
                resolve(window.twttr);
                return;
            }
            const script = document.createElement('script');
            script.src = "https://platform.twitter.com/widgets.js";
            script.async = true;
            script.onload = () => {
                resolve(window.twttr);
            };
            document.head.appendChild(script);
        });
    }

    function createToggleButton() {
        const btn = document.createElement('button');
        btn.textContent = '▽ ツイートを開く';
        btn.style.all = 'unset';
        btn.style.cursor = 'pointer';
        btn.style.color = '#4a90e2';
        btn.style.fontWeight = 'bold';
        btn.style.display = 'inline-flex';
        btn.style.alignItems = 'center';
        btn.style.userSelect = 'none';
        btn.style.margin = '4px 0'; // ボタン上下の余白は margin で調整
        btn.style.fontSize = '14px';
        btn.style.transition = 'transform 0.3s ease, background-color 0.3s ease';
        btn.style.borderBottom = '1px solid #4a90e2';
        btn.style.padding = '2px 8px';
        btn.style.borderRadius = '4px';
        btn.style.backgroundColor = 'rgba(74,144,226,0.1)';
        btn.addEventListener('mouseenter', () => {
            btn.style.backgroundColor = 'rgba(74,144,226,0.2)';
        });
        btn.addEventListener('mouseleave', () => {
            btn.style.backgroundColor = 'rgba(74,144,226,0.1)';
        });
        return btn;
    }

    async function replaceTwitterLinks(post) {
        if (post.dataset.twitterProcessed) return;
        post.dataset.twitterProcessed = 'true';

        const anchors = post.querySelectorAll('a[href]');
        let replaced = false;

        const twttr = await loadTwitterWidgets();

        for (const a of anchors) {
            let href = a.href;

            if (href.startsWith('../')) {
                href = window.location.origin + href.substring(2);
            }

            if (a.closest('[data-twitter-thumbnail]')) continue;

            const match = href.match(/^https?:\/\/(twitter\.com|x\.com)\/([^\/]+)\/status\/(\d+)/i);
            if (match && !a.closest('blockquote.twitter-tweet')) {

                // コンテナ作成
                const container = document.createElement('div');
                container.style.margin = '8px 0';

                // ボタンとツイート表示用コンテナ
                const toggleBtn = createToggleButton();
                const tweetContainer = document.createElement('div');
                tweetContainer.style.display = 'none';
                tweetContainer.style.marginTop = '4px'; // ボタンとツイート間の余白

                // ボタンイベント
                toggleBtn.addEventListener('click', () => {
                    if (tweetContainer.style.display === 'none') {
                        tweetContainer.style.display = 'block';
                        toggleBtn.textContent = '▲ ツイートを閉じる';

                        if (!tweetContainer.dataset.embedded) {
                            tweetContainer.dataset.embedded = 'true';
                            twttr.widgets.createTweetEmbed(
                                match[3],
                                tweetContainer,
                                {
                                    theme: 'light',
                                    conversation: 'none',
                                    dnt: true
                                }
                            ).catch(e => console.error('Tweet埋め込み失敗:', e));
                        }
                    } else {
                        tweetContainer.style.display = 'none';
                        toggleBtn.textContent = '▽ ツイートを開く';
                    }
                });

                container.appendChild(toggleBtn);
                container.appendChild(tweetContainer);

                a.replaceWith(container);
                replaced = true;
            }
        }

        if (replaced) {
            console.log('Twitterリンクを開閉ボタン付き公式埋め込みに置換しました（改行なし、margin調整）');
        }
    }

    // IntersectionObserverで表示範囲に入るまで待ってから埋め込み処理
    const observer = new IntersectionObserver((entries, observer) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const post = entry.target;
                replaceTwitterLinks(post);
                observer.unobserve(post); // 一度処理したら監視解除
            }
        });
    }, {
        root: null,
        rootMargin: '0px',
        threshold: 0
    });

    // ページロード時は監視登録のみ（即時処理はしない）
    document.querySelectorAll('.post-content').forEach(post => {
        observer.observe(post);
    });

    // MutationObserverは追加された要素を監視登録
    const mutationObserver = new MutationObserver(mutations => {
        for (const mutation of mutations) {
            for (const node of mutation.addedNodes) {
                if (node.nodeType !== 1) continue;
                if (node.classList.contains('post-content')) {
                    observer.observe(node);
                } else {
                    node.querySelectorAll?.('.post-content').forEach(child => observer.observe(child));
                }
            }
        }
    });

    mutationObserver.observe(document.body, {
        childList: true,
        subtree: true
    });

})();
